/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* FILE.C --- Load and Save boxes, etc. */

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <graphics.h>
#include <dos.h>
#include <dir.h>
#include <errno.h>
#include <sys\stat.h>
#include "window.h"
#include "resource.h"
#include "key.h"

/* ----- Directory fetching ----------------------------------------- */

/* Extend fnsplit flags. */
#define EXISTS  0x20
#define PATHERR 0x40
#define NAMEERR 0x80

/* Check a path for validity. If a bare directory,
   add *.<def_ext>.  If ends in *, add .<def_ext>.  
   If a bare file name, add .<def_ext> and return 
   flag showing existence (or adjusted path if 
   default extension turns up a directory).
     Return:
       PATHERR   -- Path to file/dir contains an error.
       NAMERR    -- File/dirname is malformed.
       WILDCARDS -- returned path is a pattern.
       EXISTS    -- path is a file that exists.
*/
LOCAL(int) expand_path(char *path, char *default_ext)
{
  int code;
  struct ffblk ffblk;
  struct stat statbuf;
  char tmp[MAXPATH], drive[MAXDRIVE], dir[MAXDIR], name[MAXFILE], ext[MAXEXT];

  strupr(path);

  /* Check validity up to the filename. */
  code = fnsplit(path, drive, dir, name, ext);
  fnmerge(tmp, drive, dir, "*", ".*");
  if (findfirst(tmp, &ffblk, 0) && errno == EINVAL)
    return (code | PATHERR);

  /* Fill in drive. */
  if (!(code & DRIVE)) {
    strcpy(drive, "X:");
    drive[0] = getdisk() + 'A';
  }

  /* If path has embedded wildcards or has no filename part, 
     add wildcard pattern and quit. */
  if ((code & WILDCARDS) || !(code & (FILENAME|EXTENSION))) {
    if (!(code & FILENAME))
      strcpy(name, "*");
    if (!(code & EXTENSION))
      strcpy(ext, default_ext /*".*"*/);
    fnmerge(path, drive, dir, name, ext);
    return (code | WILDCARDS);
  }

  /* Path is valid but has no wildcards.  May be
     nonexistent, erroneous name, file, or directory.
     Find out which. */
  for (;;) {
    fnmerge(path, drive, dir, name, ext);
    if (stat(path, &statbuf)) {
      /* Non-existent. */
      if (_doserrno == 3)
	return (code | NAMEERR);
      if (code & EXTENSION)
	return code;
    }
    else if (statbuf.st_mode & S_IFDIR) {
      /* Existing dir */
      strcat(path, "\\");
      code = fnsplit(path, drive, dir, NULL, NULL);
      fnmerge(path, drive, dir, "*", default_ext /*".*"*/ );
      return (code | WILDCARDS);
    }
    else if (code & EXTENSION)
      return (code | EXISTS);

    /* Exists without default ext. We don't care. Add
       extension and loop.  Second time must succeed. */
    strcpy(ext, default_ext);
    code |= EXTENSION;
  }
}

/* Call DOS function to eliminate \a\..\,
   etc. from path name. */
LOCAL(int) canonicalize_path(char *path)
{
  union REGS in, out;
  struct SREGS sregs;
  char buf[128];

  in.h.ah = 0x60;
  sregs.ds = FP_SEG(path);
  in.x.si = FP_OFF(path);
  sregs.es = FP_SEG(buf);
  in.x.di = FP_OFF(buf);
  intdosx(&in, &out, &sregs);
  if (out.x.flags & 1)
    return out.x.ax;
  else {
    strcpy(path, buf);
    return 0;
  }
}

/* Return non-0 iff `p' is a
   directory entry from get_dir_entries(). */
LOCAL(int) dirname_p(const char *p)
{
  return (p[strlen(p) - 1] == '\\');
}

/* Compare two filenames for get_dir_entries() sort. */
static int filename_cmp(const void *p0, const void *p1)
{
  int dir0_p = dirname_p(p0);
  int dir1_p = dirname_p(p1);
  if (dir0_p != dir1_p)
    return dir0_p - dir1_p;
  else
    return strcmp(p0, p1);
}

/* Free storage for a set of entries as allocated above. */
LOCAL(void) free_dir_entries(char **e)
{
  if (e != NULL) {
    free(e[0]);
    free(e);
  }
}

/* Get table entries that are sorted
   directory elements given by "path".
   Free storage in `*entries_rtn' if
   it is non-NULL. Set `entries' NULL
   if nothing found, else malloc a block
   for the answer. */
LOCAL(void) get_dir_entries(char *path, char ***entries_rtn, int *n_rtn)
{
  struct ffblk ffblk;
  struct entry { char name[sizeof ffblk.ff_name + 1]; } *e;
  int done_p, n, cur_size;
  char drive[MAXDRIVE], dir[MAXDIR], dir_path[MAXPATH];
# define DIR_ENTRIES_ALLOC_SIZE 50

  /* Free old storage (if non-NULL) */
  free_dir_entries(*entries_rtn);

  /* Get file names into a block of storage, guessing its size. */
  e = malloc(sizeof(struct entry) * DIR_ENTRIES_ALLOC_SIZE);
  cur_size = DIR_ENTRIES_ALLOC_SIZE;
  for (done_p = findfirst(path, &ffblk, 0), n = 0; !done_p; done_p = findnext(&ffblk), ++n) {
    if (n >= cur_size) {
      cur_size += DIR_ENTRIES_ALLOC_SIZE;
      e = realloc(e, sizeof(struct entry) * cur_size);
    }
    strcpy(e[n].name, ffblk.ff_name);
  }

  /* Construct path that gets all directories. */
  fnsplit(path, drive, dir, NULL, NULL);
  fnmerge(dir_path, drive, dir, "*", ".*");

  /* Tack on directory names at the end. */
  for (done_p = findfirst(dir_path, &ffblk, FA_DIREC); !done_p; done_p = findnext(&ffblk)) {
    if (!(ffblk.ff_attrib & FA_DIREC) || !strcmp(ffblk.ff_name, "."))
      continue;
    if (n >= cur_size) {
      cur_size += DIR_ENTRIES_ALLOC_SIZE;
      e = realloc(e, sizeof(struct entry) * cur_size);
    }
    strcat(strcpy(e[n].name, ffblk.ff_name), "\\");
    ++n;
  }

  /* Set up number of return entries. If non-zero, sort
     and create the array of pointers to them.  Else
     free the block of names and return null. */
  *n_rtn = n;
  if (n > 0) {
    char **r, **end;

    /* Shrink to fit. */
    e = realloc(e, sizeof(struct entry) * n);

    qsort(e, n, sizeof(struct entry), filename_cmp);

    /* Build returned array of pointers to entries. */
    for (r = *entries_rtn = malloc(n * sizeof(char*)), end = r + n;
	 r < end;
	 ++r, ++e)
      *r = e->name;
  }
  else {
    free(e);
    *entries_rtn = NULL;
  }
}

/* Change extension of filename in `path' to `.set'.
   Return the modified path in the provided buffer `result'. */
char *change_suffix(const char *path, char *sfx, char *result)
{
  char drive[MAXDRIVE], dir[MAXDIR], name[MAXFILE];

  fnsplit(path, drive, dir, name, NULL);
  fnmerge(result, drive, dir, name, sfx);
  return result;
}

/* ----- File dialogs. ---------------------------------------------- */

void handle_edit_completion(char *path);
void handle_ok_press(void);
void handle_cancel_press(void);
void handle_table_completion(char *entry);

enum { LOAD_EDIT, LOAD_TABLE, LOAD_OK, LOAD_CANCEL };

BeginDefEditBox(load_edit)
  DefKeyAction('\r', NameActionClosure(handle_edit_completion, NullEnv))
EndDefEditBox(load_edit, "Name", 0, 30, MAXPATH, Enabled)

BeginDefTable(load_table)
  DefKeyAction('\r', NameActionClosure(handle_table_completion, NullEnv))
EndDefTable(load_table, "Files", 0, NoEntries, 0, 13, 8, 8, 2, Enabled)

DefBitmapButton(load_ok,     ok,     'k', 2, 2, Enabled, ActionClosure(handle_ok_press, NullEnv))
DefBitmapButton(load_cancel, cancel, ESC, 2, 2, Enabled, ActionClosure(handle_cancel_press, NullEnv))

BeginDefDialog(load)
  DefDialogItem(EditBoxItem, load_edit, 8, 16)
  DefDialogItem(TableItem, load_table, 8, 44)
  DefDialogItem(ButtonItem, load_ok, 257, 72)
  DefDialogItem(ButtonItem, load_cancel, 242, 103)
EndDefDialog(load, "", Enabled)

static char **entries = NULL;
static char load_pattern[MAXPATH] = "*.*";
static char default_ext[MAXEXT] = ".";

/* Shut down the load dialog. */
static char *rtn_path;
static void stop_load_dialog(char *path, int code)
{
  if (path != NULL)
    strcpy(rtn_path, path);
  stop_modal_dialog(load, code);
}

/* Fill the load table with entries.  Return 1 if
   wildcards were found. Fills dir entry table if
   path contains wildcards.  Else just returns
   expanded path. */
LOCAL(int) fill_edit_table(char *path)
{
  int n, code;
  static char *no_files[] = {"No files"};

  code = expand_path(path, default_ext);
  if (code & PATHERR) {
    strcpy(&path[20], " ...");
    message(bit(mOK), &load->window, "Invalid path:\n`%s'", path);
    strcpy(path, load_pattern);
    code |= WILDCARDS;
  }
  else if (code & NAMEERR) {
    char file[MAXFILE], ext[MAXEXT];
    fnsplit(path, NULL, NULL, file, ext);
    message(bit(mOK), &load->window, "Invalid filename:\n`%s%s'", file, ext);
    strcpy(path, load_pattern);
    code |= WILDCARDS;
  }
  if (code & WILDCARDS) {
    get_dir_entries(path, &entries, &n);
    strcpy(load_pattern, path);
    if (entries == NULL) {
      disable_table(dialog_item(load, LOAD_TABLE, TABLE));
      set_table_entries(dialog_item(load, LOAD_TABLE, TABLE), no_files, 1);
    }
    else {
      set_table_entries(dialog_item(load, LOAD_TABLE, TABLE), entries, n);
      enable_table(dialog_item(load, LOAD_TABLE, TABLE));
    }
  }

  return code;
}

static void handle_edit_completion(char *path)
{
  int code = fill_edit_table(path);

  if (code & WILDCARDS) {
    set_edit_text(dialog_item(load, LOAD_EDIT, EDIT_BOX), path);
    if (entries != NULL) 
      set_focus(load->items[LOAD_TABLE].item);
  }
  else
    stop_load_dialog(path, (code & EXISTS) == 0);
}

static void handle_table_completion(char *entry)
{
  char path[MAXPATH], drive[MAXDRIVE], dir[MAXDIR+MAXFILE+1], name[MAXFILE], ext[MAXEXT];

  fnsplit(get_edit_text(dialog_item(load, LOAD_EDIT, EDIT_BOX)), drive, dir, NULL, NULL);

  if (dirname_p(entry)) {
    strcat(dir, entry);
    fnmerge(path, drive, dir, "", "");
    canonicalize_path(path);
    fill_edit_table(path);
    set_edit_text(dialog_item(load, LOAD_EDIT, EDIT_BOX), path);
  }
  else {
    fnsplit(entry, NULL, NULL, name, ext);
    fnmerge(path, drive, dir, name, ext);
    stop_load_dialog(path, 0);
  }
}

static void handle_ok_press(void)
{
  if (focus_window == load->items[LOAD_EDIT].item)
    handle_edit_completion(get_edit_text(dialog_item(load, LOAD_EDIT, EDIT_BOX)));
  else
    /* This is executed on mouse click or 
       enter/space when ok button has keyboard focus. */
    handle_table_completion(load_table->entries[load_table->pos]);
}

static void handle_cancel_press(void) { stop_load_dialog(NULL, 2); }

int get_path(char *title, char *path, char *ext)
{
  if (!static_resource_opened_p(load)) 
    open_dialog(load, CENTER_WINDOW, CENTER_WINDOW, root_window);

  /* On empty path, fill table with all names having default 
     extension.  This is for selecting a save filename. */
  if (path[0] == '\0') {
    fill_edit_table(strcat(strcpy(path, "*"), ext));
    path[0] = '\0';
  }
  else
    fill_edit_table(path);

  set_edit_text(dialog_item(load, LOAD_EDIT, EDIT_BOX), path);

  load->title = title;
  rtn_path = path;
  strcpy(default_ext, ext);
  load_pattern[0] = '*'; strcpy(&load_pattern[1], ext);
  return run_modal_dialog(load);
}
