## -*-Tcl-*-
 # ###################################################################
 #  Vince's Additions - an extension package for Alpha
 # 
 #  FILE: "alphaHooks.tcl"
 #                                    created: 18/7/97 {5:10:18 pm} 
 #                                last update: 1999-09-04T14:01:03Z 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 1997-1999  Vince Darley, all rights reserved
 #  
 # Description: 
 #  
 #  Here are the current hooks:
 #  
 #  activateHook changeMode closeHook deactivateHook modifyModeFlags 
 #  quitHook resumeHook saveasHook saveHook savePostHook suspendHook
 #  openHook
 #  
 #  There's also a 'mode::init' hook which will be called the first
 #  time a mode is started up.  Note that the mode exists, but its
 #  variables have not yet been made global, and its menus have not
 #  yet been inserted into the menu bar.
 #  
 #  There's also a 'startupHook' which is called when Alpha starts
 #  up, but after all other initialisation has taken place (before
 #  any files are opened though).
 #  
 #  There's also a 'launch' hook for when an app is launched.
 #  
 #  Use of such lists as 'savePostHooks' is obsolete.
 #  These lists are ignored, use hook::register instead.
 #  
 #  History
 # 
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  18/7/97  VMD 1.0 original
 #  22/7/97  VMD 1.1 fixed all bugs ;-) and added the above examples.
 # ###################################################################
 ##

namespace eval mode {}
namespace eval win {}

lappend mode::procs carriageReturn OptionTitleBar OptionTitleBarSelect \
  electricLeft electricRight electricSemi indentLine indentRegion \
  parseFuncs MarkFile

proc saveHook name {
    global backup backupExtension backupFolder mode win::Modes \
      backupAgeRequirementInHours modifiedVars
    hook::callAll saveHook [set win::Modes($name)] $name
    if {![file exists $name]} {
	regsub { <[0-9]+>$} $name {} name
    }
    if {$backup} {
	if {$backupFolder != "" && ![file exists $backupFolder]} {
	    if {![dialog::yesno "Create backup folder '$backupFolder'?"]} {
		alertnote "Backup saved in document's folder."
		set backupFolder ""
		lappend modifiedVars backupFolder
	    } elseif {[catch {file::ensureDirExists $backupFolder}]} {
		alertnote "Couldn't create backup folder. Backup saved in document's folder."
		set backupFolder ""
		lappend modifiedVars backupFolder
	    }
	}
	set dir $backupFolder

	if {![string length $dir]} {
	    set dir [file dirname $name]
	}
	if {$backupExtension == "" && $backupFolder == ""} {
	    set backupExtension ~
	    lappend modifiedVars backupExtension
	}
	set backfile [file join $dir [file tail $name]$backupExtension]
	if {$backupExtension == "" && [file dirname $name] == $backupFolder} {
	    append backfile ~
	}
	if {[file exists $backfile]} {
	    getFileInfo $name a
	    if {[expr {([now] - $a(modified) + 0.0)/3600}] < $backupAgeRequirementInHours} {
		return
	    }
	    catch {file delete $backfile}
	}
	message "Backing up $backfile"
	catch {file copy $name $backfile}
    }

}

proc saveUnmodified {} {
    set name [win::Current]
    if {[file exists $name] || \
      ([regsub { <\w+>$} $name {} name] && [file exists $name])} {
	getFileInfo $name arr
	set mod $arr(modified)
	save
	setFileInfo $name modified $mod
	return
    }
    # shouldn't really get here!
    error "File doesn't exist"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "changeMode" --
 # 
 #  A very important procedure.  It handles all switching from one mode
 #  to another.  This means it has to adjust menus, floating windows,
 #  global variables, mode prefs, and call a number of hooks.
 #  
 #  It maintains a list of variables which the new mode over-rides from
 #  the global scope, and recreates them.  This allows a mode to have
 #  its own value for a global variable without messing anything up.
 # -------------------------------------------------------------------------
 ##
proc changeMode {newMode} {
    global lastMode dummyProc mode seenMode PREFS global::_varMem \
      mode::features global::features global::_oldTabSize
	
    # This section should restore any internally shadowed globals, 
    # currently only tabSize may be stored by 'new'
    if {[info exists global::_oldTabSize]} {
	global tabSize
	set tabSize [set global::_oldTabSize]
	unset global::_oldTabSize
    }
    
    set lastMode $mode
    set mode $newMode
    if {$lastMode == $mode} {
        if {$newMode != ""} {
	    displayMode $newMode
	}
        return
    }
    if {$lastMode == ""} {
	renameMenuItem -m Config "Mode Prefs" "${mode} Mode Prefs"
	catch {menuEnableHook 1}
    } elseif {$mode == ""} {
	renameMenuItem -m Config "${lastMode} Mode Prefs" "Mode Prefs"
	catch {menuEnableHook 0}
    } else {
	renameMenuItem -m Config "${lastMode} Mode Prefs" "${mode} Mode Prefs"
    }
    
    # Get rid of all the old mode's variables, but only if it is necessary
    # (Else we screw up traces on those variables)
    global ${lastMode}modeVars
    if {[info exists ${lastMode}modeVars]} {
        foreach v [array names ${lastMode}modeVars] {
	    if {![info exists global::_varMem($v)]} {
		global $v
		catch {unset $v}
	    }
        }
    }
    floatShowHide off $lastMode
    if {[info exists global::_varMem]} {
	foreach v [array names global::_varMem] {
	    global $v
	    set $v [set global::_varMem($v)]
	}
	unset global::_varMem
    }
    if {[info exists mode::features($mode)]} {
	set onoff [package::onOrOff [set mode::features($mode)] $lastMode]
    } else {
	set onoff [package::onOrOff "" $lastMode]
    }
    
    foreach m [lindex $onoff 0] {
	package::deactivate $m
    }
    
    # These lines must load the mode vars into the mode var scope.
    if {[info exists dummyProc($mode)]} { $dummyProc($mode) }
    if {![info exists seenMode($mode)]} {
	hook::callAll mode::init $mode
    }
    # once the vars are in mode-var scope (= the <mode>modeVars array),
    # they can be transfered to the global scope.  A future version of
    # Alpha with Tcl8.0 namespaces may not need to do this.
    global ${mode}modeVars
    if {[info exists ${mode}modeVars]} {
        foreach v [array names ${mode}modeVars] {
            global $v
	    if {[info exists $v]} { 
		set global::_varMem($v) [set $v]
	    }
            set $v [set ${mode}modeVars($v)]
        }
    }
    foreach m [lindex $onoff 1] {
	package::activate $m
    }
    
    floatShowHide on $mode

    if {![info exists seenMode($mode)]} {
	global mode::procs
	#foreach p ${mode::procs} {
	#    if {[info commands ${mode}::${p}] == ""} {
	#	auto_load ${mode}::${p}
	#    }
	#}
	set seenMode($mode) 1
	if {($mode != "") && [file exists [file join $PREFS ${mode}Prefs.tcl]]} {
	    if {[catch {uplevel \#0 [list source [file join $PREFS ${mode}Prefs.tcl]]}]} {
                alertnote "Your preferences file '${mode}Prefs.tcl has an error."
            } 
        }
    }
        
    if {$newMode != ""} {
	displayMode $newMode
    }

    hook::callAll changeMode $mode $mode
}

## 
 # -------------------------------------------------------------------------
 # 
 # "requireOpenWindowsHook" --
 # 
 #  En-/disable meaningless menu items which would require the presence
 #  of a certain number of windows to be active
 #  
 #  This proc should only be called from 'openHook' and 'closeHook'.
 #  
 #  You can register with it using 
 #  
 #  'hook::register requireOpenWindowsHook [list menu item] N'
 #  
 #  where 'N' is the number of windows required (1 or 2 usually)
 #  (and deregister etc using hook::deregister).
 #  
 #  We only really need the catch in here for two reasons:
 #  (i) in case bad menus are registered accidentally
 #  (ii) so startup errors can open a window without hitting another error
 #  in the middle of doing that!
 # -------------------------------------------------------------------------
 ##
proc requireOpenWindowsHook {requiredNum} {
    global win::Active
    foreach count $requiredNum {
	set enable [expr {[llength [set win::Active]] >= $requiredNum ? 1 : 0}]
	foreach i [hook::information requireOpenWindowsHook $requiredNum] {
	    catch "enableMenuItem $i $enable"
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menuEnableHook" --
 # 
 #  This hook is called to turn menu items on or off.  It is called 
 #  whenever there are no windows, or when we go from 0->1 window.
 #  
 #  It should deal with all standard menus.  It does not deal with
 #  special menu items like 'save', 'revert',.. which require more
 #  information.
 #  
 #  It is called from changeMode.
 #  
 #  Andreas wrote most of this proc.
 #  
 #  Due to a deficiency in MacOS/MercutioMDEF/Alpha (not sure who
 #  the culprit is!), key-bindings attached to menu items are still
 #  triggered even if the menu item is inactive.
 # -------------------------------------------------------------------------
 ##
proc menuEnableHook {{haveWin 1}} {
    global winMenu mode
    # we only get here if there are no windows, or 1 window which we
    # just opened.  Otherwise nothing will be different to last time.
    enableMenuItem File close $haveWin
    enableMenuItem File closeAll $haveWin
    enableMenuItem File closeFloat $haveWin
    enableMenuItem File saveAs $haveWin
    enableMenuItem File saveACopyAs $haveWin
    if {[package::active printerChoicesMenu]} {
	enableMenuItem File print $haveWin
    } else {
	enableMenuItem File print $haveWin
    }
    enableMenuItem File printAll $haveWin
    eval [lindex [list un {}] $haveWin]Bind 'p' <c> print
    
    enableMenuItem Edit undo $haveWin
    enableMenuItem Edit redo $haveWin
    enableMenuItem Edit cut $haveWin
    enableMenuItem Edit copy $haveWin
    enableMenuItem Edit paste $haveWin
    enableMenuItem Edit selectAll $haveWin
    enableMenuItem Edit selectParagraph $haveWin
    enableMenuItem Edit clear $haveWin
    enableMenuItem Edit twiddle $haveWin
    enableMenuItem Edit twiddleWords $haveWin
    enableMenuItem Edit shiftLeft  $haveWin
    enableMenuItem Edit shiftLeftSpace  $haveWin
    enableMenuItem Edit shiftRight  $haveWin
    enableMenuItem Edit shiftRightSpace  $haveWin
    enableMenuItem Edit balance  $haveWin

    if {[info tclversion] < 8.0} {
        enableMenuItem Text fillParagraph $haveWin
        enableMenuItem Text wrapParagraph $haveWin
        enableMenuItem Text sentenceParagraph $haveWin
        enableMenuItem Text fillRegion $haveWin
        enableMenuItem Text wrapRegion $haveWin
        enableMenuItem Text sentenceRegion $haveWin
        enableMenuItem Text paragraphToLine $haveWin
        enableMenuItem Text lineToParagraph $haveWin
        enableMenuItem Text reverseSort $haveWin
        enableMenuItem Text sortLines $haveWin
        enableMenuItem Text sortParagraphs $haveWin
        enableMenuItem Text zapInvisibles $haveWin
        enableMenuItem Text tabsToSpaces $haveWin
        enableMenuItem Text spacesToTabs $haveWin
        enableMenuItem Text indentLine $haveWin
        enableMenuItem Text indentSelection $haveWin
        enableMenuItem Text upcaseRegion $haveWin
        enableMenuItem Text downcaseRegion $haveWin
        enableMenuItem Text strings $haveWin
        enableMenuItem Text commentLine $haveWin
        enableMenuItem Text uncommentLine $haveWin
        enableMenuItem Text commentBox $haveWin
        enableMenuItem Text uncommentBox $haveWin
        enableMenuItem Text commentParagraph $haveWin
        enableMenuItem Text uncommentParagraph $haveWin
	enableMenuItem Config "Mode Prefs" $haveWin
    } else {
	enableMenuItem Text "" $haveWin
	if {$mode == ""} {
	    enableMenuItem -m Config "Mode Prefs" $haveWin
	} else {
	    enableMenuItem -m Config "${mode} Mode Prefs" $haveWin
	}
    }
    
    enableMenuItem Search searchStart $haveWin
    enableMenuItem Search findAgain $haveWin
    enableMenuItem Search findAgainBackward $haveWin
    if { ![string compare [searchString] ""] && !$haveWin } {
	enableMenuItem Search findInNextFile $haveWin
    } else {
	enableMenuItem Search findInNextFile 1
    }
    enableMenuItem Search enterSearchString $haveWin
    enableMenuItem Search enterReplaceString $haveWin
    enableMenuItem Search quickFind $haveWin
    enableMenuItem Search quickFindRegexp $haveWin
    enableMenuItem Search reverseQuickFind $haveWin
    enableMenuItem Search replace $haveWin
    enableMenuItem Search replace&FindAgain $haveWin
    enableMenuItem Search replaceAll $haveWin
    enableMenuItem Search placeBookmark $haveWin
    enableMenuItem Search returnToBookmark $haveWin
    enableMenuItem Search gotoLine $haveWin
    enableMenuItem Search matchingLines $haveWin
    enableMenuItem Search gotoMatch $haveWin
    enableMenuItem Search nextMatch $haveWin
    enableMenuItem Search gotoFunc $haveWin
    enableMenuItem Search gotoFileMark $haveWin
    enableMenuItem Search markHilite $haveWin
    enableMenuItem Search namedMarks $haveWin
    enableMenuItem Search unnamedMarks $haveWin
    
    enableMenuItem Utils AsciiEtc $haveWin
    enableMenuItem Utils cmdDoubleClick $haveWin
    enableMenuItem Utils winUtils $haveWin
    enableMenuItem Utils spellcheckWindow $haveWin
    enableMenuItem Utils spellcheckSelection $haveWin
    enableMenuItem Utils wordCount $haveWin
    
    enableMenuItem Config setFontsTabs $haveWin
    
    enableMenuItem $winMenu zoom $haveWin
    enableMenuItem $winMenu defaultSize $haveWin
    enableMenuItem $winMenu chooseAWindow $haveWin
    enableMenuItem $winMenu iconify $haveWin
    enableMenuItem $winMenu arrange $haveWin
    enableMenuItem $winMenu splitWindow $haveWin
    enableMenuItem $winMenu toggleScrollbar $haveWin
    
    if {!$haveWin} {
	enableMenuItem File save 0
	enableMenuItem File saveUnmodified 0
	enableMenuItem File revert 0
	enableMenuItem File revertToBackup 0
	enableMenuItem File renameTo 0
	enableMenuItem File saveAll 0
    }
    
    requireOpenWindowsHook 1
}

proc savePostHook name {
    # So modified date is  ok
    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
      && (![catch {getFileInfo $nm info}]))} {
	global win::Modified
	set win::Modified($name) $info(modified)
    } else {
	if {[info tclversion] < 8.0} {
	    # Alpha bug workaround
	    set name [subst $name]
	    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
	      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
	      && (![catch {getFileInfo $nm info}]))} {
		global win::Modified
		set win::Modified($name) $info(modified)
	    } else {
		alertnote "Weird, file '$name' doesn't seem to exist: please\
		  report the circumstances of this problem to the Alpha-D mailing list."
	    }
	} else {
	    alertnote "Weird, file '$name' doesn't seem to exist: please\
	      report the circumstances of this problem to the Alpha-D mailing list."
	}
    }
    hook::callAll savePostHook "" $name
}

proc closeHook name {
    global markStack win::Modes win::Active win::Current win::Dirty \
      win::NumDirty win::Modified
    hook::callAll closeHook [set win::Modes($name)] $name

    if {[info exists win::Dirty($name)]} {
	incr win::NumDirty -1
	unset win::Dirty($name)
	enableMenuItem File saveAll [expr {${win::NumDirty} ? 1 : 0}]
    }
		
    unset win::Modes($name)
    if {[info exists win::Modified($name)]} {
	unset win::Modified($name)
    }
    
    if {[llength $markStack]} {
        set markStack [lremove -glob $markStack $name*]
    }
    win::removeFromMenu $name

    if {[set ind [lsearch -exact ${win::Active} $name]] >= 0} {
        set win::Active [lreplace ${win::Active} $ind $ind]
    }
    if {![llength [winNames]]} {
	set win::Current ""
	changeMode {}
    }
    requireOpenWindowsHook 2
}

proc deactivateHook name {
    hook::callAll deactivateHook "" $name
}

proc suspendHook name {
    hook::callAll suspendHook "" $name
    global iconifyOnSwitch
    global suspIconed
    if {$iconifyOnSwitch} {
        set wins [winNames -f]
        set suspIconed ""
        foreach win $wins {
            if {![icon -f "$win" -q]} {
                lappend suspIconed $win
                icon -f "$win" -t
            }
        }
        set suspIconed [lreverse $suspIconed]
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "resumeHook" --
 # 
 #  The parameter 'name' is not used, so please ignore it.
 # -------------------------------------------------------------------------
 ##
proc resumeHook name {
    global iconifyOnSwitch resumeRevert suspIconed killCompilerErrors
    
    if {[info exists killCompilerErrors] && $killCompilerErrors} {
	set wins [winNames -f]
	if {[set res [lsearch $wins "*Compiler Errors*"]] >= 0} {
	    bringToFront [lindex $wins $res]
	    killWindow
	}
    }
    
    if {$iconifyOnSwitch && [info exists suspIconed]} {
	set wins [winNames -f]
	foreach win $suspIconed {
	    icon -f "$win" -o
	}
	unset suspIconed
    }
    if {$resumeRevert} {
	set resumeRevert 0
	revert
    }
    # Check if the foremost window needs to be have its modified
    # status adjusted
    modifiedCheck [win::Current]

    hook::callAll resumeHook "" $name
}

proc modifiedCheck {name} {
    if {$name != ""} {
	if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
	  ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
	  && (![catch {getFileInfo $nm info}]))} {
	    if {[catch {getWinInfo -w $name arr}]} {
		set mod 0
	    } else {
		set dirty $arr(dirty)
		if {!$dirty} {
		    global win::Modified
		    set mod [expr {[set win::Modified($name)] < $info(modified)}]
		    if {$mod} { 
			diskModifiedHook $name
		    }
		} else {
		    set mod 1
		}
	    }
	    enableMenuItem File save $mod
	    enableMenuItem File revert $mod
	}
    }
}

proc diskModifiedHook {name} {
    message "File has changed on disk since last save."
    hook::callAll diskModifiedHook
}

## 
 # -------------------------------------------------------------------------
 # 
 # "saveasHook" --
 # 
 #  Called when saving a window which doesn't yet exist as a file
 #  (in particular 'Untitled' windows) or when the user selects
 #  saveAs.
 # -------------------------------------------------------------------------
 ##
proc saveasHook {oldName newName} {
    global win::Modes win::Active win::Current win::Modified
    if {$oldName == $newName} return
    win::removeFromMenu $oldName
    win::addToMenu $newName
    win::setMode $newName
    changeMode [set win::Modes($newName)]
    
    if {[set ind [lsearch -exact ${win::Active} $oldName]] >= 0} {
	set win::Active [linsert [lreplace ${win::Active} $ind $ind] 0 $newName]
    } else {
	# hmmm! this is bad.  The old window has gone!
	set win::Active [linsert ${win::Active} 0 $newName]
    }
    
    set win::Current $newName
    if {[info exists win::Modes($oldName)]} {
	unset win::Modes($oldName)
    }
    if {[info exists win::Modified($oldName)]} {
	unset win::Modified($oldName)
    }

    hook::callAll saveasHook [set win::Modes($newName)] $oldName $newName
    refresh
}

## 
 # -------------------------------------------------------------------------
 # 
 # "saveACopyAs" --
 # 
 # Finally a proc to add to your collection of Alpha bugs.
 # copyFile has an interesting bug. If the destination file exists it
 # puts the file in [pwd] instead. This proc makes sure it is removed first.
 #  
 # (This proc actually has nothing to do with hooks, but seemed to fit here)
 # -------------------------------------------------------------------------
 ##
proc saveACopyAs {} {
    if {[file exists [set nm [stripNameCount [win::Current]]]]} {
	set nm2 [putfile "Save a copy as:" [file tail $nm]]
	if {[file exists $nm2]} {file delete $nm2}
	file copy $nm $nm2
    }
}


ensureset win::Active ""

proc activateHook {name} {
    global win::Modes win::Active win::Current win::Modified
    
    if {![info exists win::Modes($name)]} {
	win::setMode $name
    }
    if {[set ind [lsearch -exact ${win::Active} $name]] == -1} {
	set win::Active [linsert ${win::Active} 0 $name]
    } elseif {$ind >= 1} {
	set win::Active [linsert [lreplace ${win::Active} $ind $ind] 0 $name]
    }
    set win::Current $name
    
    changeMode [set win::Modes($name)]
    
    hook::callAll activateHook [set win::Modes($name)] $name
    # if the file exists (this seems to be the quickest way to check)
    if {[file exists $name] || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm])} {
	# this fails if the window is just opening, but then we know it's clean
	if {[catch {getWinInfo -w $name arr}]} {
	    set dirty 0
	    set mod 0
	} else {
	    set dirty $arr(dirty)
	    if {!$dirty} {
		if {[info exists win::Modified($name)]} {
		    if {[info exists nm]} {
			getFileInfo $nm modarr
		    } else {
			getFileInfo $name modarr
		    }
		    set mod [expr {[set win::Modified($name)] < $modarr(modified)}]
		    if {$mod} { 
			diskModifiedHook $name
		    }
		} else {
		    # This is only reached if the window is just opening,
		    # and if we fix getWinInfo not to 'catch' above.
		    set mod 0
		}
	    } else {
		set mod 1
	    }
	}
	enableMenuItem File save $mod
	enableMenuItem File saveUnmodified $dirty
	enableMenuItem File revert $mod
	enableMenuItem File revertToBackup 1
	enableMenuItem File renameTo 1
	enableMenuItem Edit undo $dirty
    } else {
	enableMenuItem File save 0
	enableMenuItem File saveUnmodified 0
	enableMenuItem File revert 0
	enableMenuItem File revertToBackup 0
	enableMenuItem File renameTo 0
	enableMenuItem Edit undo 0
    }
    
}

proc quitHook {} {
    global PREFS alpha::tracingChannel
    if {[file exists [file join $PREFS ftpTmp]]} {
        catch {rm [file join $PREFS ftpTmp *]}
    }
    catch {close ${alpha::tracingChannel}}
    saveModifiedVars
    hook::callAll quitHook
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dirtyHook" --
 # 
 #  This proc currently has to keep track in the array 'win::Dirty' of
 #  the dirty status of windows.  Its only use is if we close a dirty
 #  window and select 'discard', we would otherwise have a faulty
 #  'win::NumDirty' count.  If there's a different solution we should
 #  get rid of the win::Dirty array.
 #  
 #  Note: closeHook is called after the window is gone, and killWindow
 #  isn't called if you click in the close-box, so they don't solve
 #  the problem.
 # -------------------------------------------------------------------------
 ##
proc dirtyHook {name dirty} {
    global winMenu win::NumDirty win::Dirty
    markMenuItem -m $winMenu [file tail $name] $dirty ""
    if {$dirty == "on" || $dirty == 1} {
	set win::Dirty($name) 1
	incr win::NumDirty 1
    } else {
	if {[info exists win::Dirty($name)]} {
	    unset win::Dirty($name)
	}
	incr win::NumDirty -1
    }
    enableMenuItem File save $dirty
    enableMenuItem File saveUnmodified $dirty
    enableMenuItem File revert $dirty
    enableMenuItem File saveAll [expr {${win::NumDirty} ? 1 : 0}]
    # we may still revertToBackup even if the file is clean.
    # however we can't just revert.
    enableMenuItem Edit undo $dirty
}

proc openHook name {
    global win::Modes autoMark mode screenHeight screenWidth \
      forceMainScreen win::Modified PREFS file::config

    changeMode [set win::Modes($name)]
    win::addToMenu $name
    message ""

    if {[info exists file::config($name)]} {
	foreach opt [set file::config($name)] {
	    catch {eval setWinInfo $opt}
	}
	unset file::config($name)
    }
    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
      && (![catch {getFileInfo $nm info}]))} {
        if {[info exists info(creator)] && ($info(creator) == {ttxt})} {
            setWinInfo dirty 0
        }
        if {[info exists info(type)] && ($info(type) == {ttro})} {
            catch {setWinInfo read-only 1}
            message "Read-only!"
        }
	set win::Modified($name) $info(modified)
    }
    
    global ${mode}modeVars
    
    if {$forceMainScreen} {
        set geo [getGeometry]
        set l [lindex $geo 0]; set t [lindex $geo 1]; set w [lindex $geo 2]; set h [lindex $geo 3]; 
        if {($l < 0) || ($t < 35) || ([expr {$l + $w}] > $screenWidth) || ([expr {$t + $h + 18}] > $screenHeight)} {
	    defaultSize
        }
    }
    getWinInfo arr
    if {!$arr(read-only)} {
	if {[info exists ${mode}modeVars(autoMark)] \
	  && [set ${mode}modeVars(autoMark)] \
	  && ![llength [getNamedMarks -n]]} {
	    markFile
	}
    }
    
    if {[string match "${PREFS}*defs.tcl" $name]} {setWinInfo read-only 1}
	
    requireOpenWindowsHook 2
    
    hook::callAll openHook [set win::Modes($name)] $name
}

## 
 # -------------------------------------------------------------------------
 # 
 # "fileMovedHook" --
 # 
 #  Called by Alpha when a window's file has been moved behind our back.
 #  (Only for Alpha using Tcl 8.0)
 # -------------------------------------------------------------------------
 ##
proc fileMovedHook {from to} {
    global win::Active winNumToName winNameToNum win::Modes win::Modified
    if {[info exists winNameToNum($from)]} {
	set i $winNameToNum($from)
	unset winNameToNum($from)
	set winNumToName($i) $to
	set winNameToNum($to) $i
    } else {
	alertnote "Can't find old window.  Bad error."
    }
    set win::Modes($to) [set win::Modes($from)]
    set win::Modified($to) [set win::Modified($from)]
    unset win::Modes($from)
    unset win::Modified($from)
    set idx [lsearch -exact ${win::Active} $from]
    if {$idx >= 0} {
	set win::Active [lreplace ${win::Active} $idx $idx $to]
    } else {
	alertnote "Can't find the old window! Bad error in fileMovedHook."
    }
    hook::callAll fileMovedHook $from $to
}

proc windowChangedHook {name} {
    global win::Modes
    hook::callAll windowChangedHook [set win::Modes($name)] $name
}

proc revertHook {name} {
    global win::Modified
    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
      && (![catch {getFileInfo $nm info}]))} {
	set win::Modified($name) $info(modified)
    }
    enableMenuItem File save 0
    enableMenuItem File revert 0
}

proc revertToBackup {} {
    global backup backupExtension backupFolder win::Modes

    set fname [stripNameCount [win::Current]]
    set dir $backupFolder
    if {$dir == ""} {
        set dir [file dirname $fname]
    }
    set bname [file join $dir "[file tail $fname]$backupExtension"]
    if {![file exists $bname]} {
        beep
        message "Backup file '$bname' does not exist"
        return
    }

    if {[dialog::yesno "Revert to backup dated '[join [mtime [file mtime $bname]]]'?"]} {
        killWindow

        edit $bname
        saveAs -f $fname
    }
}







