## -*-Tcl-*- (install)
 # ###################################################################
 # 
 #  FILE: "latexAccents.tcl"
 #                                    created: 14/1/98 {6:57:41 pm} 
 #                                last update: 7/5/98 {8:10:52 pm} 
 # Version:      1.03
 # Purpose:      Translates Mac characters to their LaTeX equivalents 
 #	       and vice-versa
 # Last Change:  1.5.97
 # Language:     TCL
 # Author:       F. Miguel Dionsio
 # Email:	fmd@math.ist.utl.pt
 # Copying:      Free
 # 
 # The purpose of this tcl script is to translate mac characters to 
 # their LaTeX equivalents and vice-versa (from within the Mac 
 # editor Alpha).  The characters are those of the Courier font, 
 # with the exception of the ones that, in LaTeX, belong to math 
 # mode (see list below).  This is useful to  people who share 
 # their LaTeX sources with non-Mac users.
 # 
 # Two procedures are defined: accentsToLatexInRegion and 
 # latexToAccentsInRegion that convert all SELECTED text.  The 
 # conversion latexToAccentsInRegion tries to match (using regular 
 # expressions) various forms of the character definition in LaTeX.
 #
 ######################################################################
 #
 # List of characters (use Courier to see them):
 # 
 #		""	""	""	""	""	 ""
 #		""	""	""	""	""	 ""
 #		""	""	""	""
 #		""	""	""	""
 #		""	""	""	""
 #		""	""	""	""
 #		""	""	""	""	""
 #		""	""	""	""	""
 #		""	""	""	""
 #		""	""	""	""
 #		""	""	""	""
 #		""	""	""	""
 #		""	""	""
 #		""	""	""	""	""	""	""	""
 #		""	""	""
 #		""	""
 # 
 #  WARNING:  appears different in Monaco
 # 
 #  List of OMITTED characters from Courier:
 # 
 #  ,,, , , and the following symbols from mathematical mode:
 #  ,,,,,,,,,,,,,,,,,,,,,,
 # 
 # If you find some more problems please tell me. (fmd@math.ist.utl.pt)
 #
 ######################################################################
 # 
 # Alpha commands used: getText, getPos, selEnd, replaceText; 
 # all others are TCL built-ins
 # 
 # ###################################################################
 ##

# extension declaration
alpha::feature latexAccents 1.0.3 {TeX Bib} {
	alpha::package require TeX 4.2
    menu::insert {LaTeX Utilities} items 5 \
	  "(-" {Convert Accents to LaTeX} {Convert LaTeX to Accents}
} "" "" maintainer {
    {F. Miguel Dionsio} fmd@math.ist.utl.pt
} help {
	Converts between international symbols and LaTeX codes
} uninstall this-file

 ######################################################################
 #
 # Returns, for argument "a" the regular expression 
 # [ \t]*(a|{[ \t]*a[ \t]*}),
 # used to look for alternative ways of writting, for example :
 # \`a, \` a, \`{a}, etc.
 #
 ######################################################################
proc rexp  {c} {
	set ws {[ \t]*}
	return $ws\($c|{$ws$c$ws}\)
}

######################################################################
# 
# This does the rest: defines the list of chars (all), the list of their 
# LaTex equivalent (texall) and the list of corresponding regular 
# expressions (regall).  When translating to LaTeX replaces all 
# ocurrences of each char by the corresponding LaTeX equivalent (using 
# regsub, see the TCL manual).  In the other direction replaces the text 
# that matches the regular expression by the corresponding char.
# 
######################################################################

proc TeX::ConvertAccentstoLaTeX {} {TeX::replaceAccents 1}
proc TeX::ConvertLaTeXtoAccents {} {TeX::replaceAccents 0}

proc TeX::replaceAccents {ww} {
	if { [isSelection] } {
		set position [getPos]
		set endselection [selEnd]
	} else {
		switch [askyesno "Convert the entire document?"] {
			"yes" {
				set position [minPos]
				set endselection [maxPos]
			}
			"no" { return }
		}
	}
	set text [getText $position $endselection]
	
	############################################################
	# 
	# List of characters to be replaced by LaTeX equivalent
	# 
	############################################################

	set as    [list "" "" "" "" ""  ""] 
	set cas   [list "" "" "" "" ""  ""] 
	set es    [list "" "" "" ""]
	set ces   [list "" "" "" ""]
	set is    [list "" "" "" ""]
	set cis   [list "" "" "" ""]
	set os    [list "" "" "" "" ""]
	set cos   [list "" "" "" "" ""]
	set us    [list "" "" "" ""]
	set cus   [list "" "" "" ""]
	set lig   [list "" "" "" ""]
	set oth1  [list "" "" "" ""]
	set coth1 [list "" "" ""]
	set oth2  [list "" "" "" "" "" "" "" ""]
	set oth3  [list "" "" ""]
	set oth4  [list "" ""]
	
	set all [concat $as $cas $es $ces $is $cis $os $cos $us $cus $lig $oth1 $coth1 $oth2 $oth3 $oth4]

	############################################################
	# 
	# List of the LaTex equivalents
	# 
	############################################################

	set quote {"}  
	set seplater {\\\\sepsep//}
      
	if {$ww == "1"} {      
		set texas    [list  "\\`{a}" "\\'{a}" "\\^{a}" "\\~{a}" "\\$quote{a}" "\\aa$seplater"] 
		set texcas   [list  "\\`{A}" "\\'{A}" "\\^{A}" "\\~{A}" "\\$quote{A}" "\\AA$seplater"]                                     
		set texes    [list  "\\`{e}" "\\'{e}" "\\^{e}" "\\$quote{e}"]                
		set texces   [list  "\\`{E}" "\\'{E}" "\\^{E}" "\\$quote{E}"]   
		set texis    [list  "\\`{\\i}" "\\'{\\i}" "\\^{\\i}"  "\\$quote{\\i}"]
		set texcis   [list  "\\`{I}"   "\\'{I}"   "\\^{I}" "\\$quote{I}"]
		set texos    [list  "\\`{o}" "\\'{o}" "\\^{o}" "\\~{o}" "\\$quote{o}"]
		set texcos   [list  "\\`{O}" "\\'{O}" "\\^{O}" "\\~{O}" "\\$quote{O}"]
		set texus    [list  "\\`{u}" "\\'{u}" "\\^{u}" "\\$quote{u}"]
		set texcus   [list  "\\`{U}" "\\'{U}" "\\^{U}" "\\$quote{U}"]
		set texlig   [list  "\\oe$seplater"  "\\OE$seplater" "\\ae$seplater" "\\AE$seplater"]
		set texoth1  [list  "\\c{c}" "\\~{n}" "\\$quote{y}" "\\dots$seplater"]
		set texcoth1 [list  "\\c{C}" "\\~{N}" "\\$quote{Y}" ]
		set texoth2  [list  "\\ss$seplater"  "?`" "!`" "\\pounds$seplater" "\\o$seplater" "\\O$seplater" "\\S$seplater" "\\P$seplater"]
		set texoth3  [list   "{\\leavevmode\\raise.585ex\\hbox{\\b{a}}}"  "{\\leavevmode\\raise.6ex\\hbox{\\b{o}}}" "\\accent'27"]
		set texoth4  [list     "\\copyright$seplater" "\\ddag$seplater"]
		    
		set texall   [concat  $texas $texcas $texes $texces $texis $texcis $texos $texcos  $texus $texcus $texlig $texoth1 $texcoth1 $texoth2 $texoth3 $texoth4]
	}

	      
	############################################################
	#														                               #
	#				 List of regular expressions		                   #
	#			     For 	the	reg	exp	is				                   #
	#			      \`[ \t]*(a|{[	\t]*a[ \t]*})		                 #
	#                 \c c needs the space but \c{c} does not  #
	#                 \i may have trailing spaces              #
	#														                               #
	############################################################

	if {$ww == "0"} {      
		set ws {[ \t]*}
		set sp {[ \t]}
		set sep { *( |\b)}
		set a [rexp a]
		set regas  [list  "\\\\`$a" "\\\\'$a" "\\\\\\^$a"  "\\\\~$a"  "\\\\\\$quote$a" "\\\\aa$sep"]
		set a [rexp A]
		set regcas [list  "\\\\`$a" "\\\\'$a" "\\\\\\^$a"  "\\\\~$a"  "\\\\\\$quote$a" "\\\\AA$sep"]
		set e [rexp e]
		set reges  [list  "\\\\`$e"  "\\\\'$e" "\\\\\\^$e"  "\\\\\\$quote$e"]
		set e [rexp E]
		set regces [list  "\\\\`$e"  "\\\\'$e" "\\\\\\^$e"  "\\\\\\$quote$e"]
		set i [rexp \\i$ws]
		set regis  [list  "\\\\`$i" "\\\\'$i"  "\\\\\\^$i"  "\\\\$quote$i"]
		set i [rexp I]
		set regcis  [list "\\\\`$i" "\\\\'$i"  "\\\\\\^$i"  "\\\\$quote$i"]
		set o [rexp o]
		set regos   [list "\\\\`$o" "\\\\'$o"  "\\\\\\^$o"  "\\\\~$o"  "\\\\$quote$o"]
		set o [rexp O]
		set regcos   [list "\\\\`$o" "\\\\'$o"  "\\\\\\^$o"  "\\\\~$o"  "\\\\$quote$o"]
		set u [rexp u]
		set regus    [list "\\\\`$u" "\\\\'$u"  "\\\\\\^$u"  "\\\\$quote$u"]
		set u [rexp U]
		set regcus   [list "\\\\`$u" "\\\\'$u"  "\\\\\\^$u"  "\\\\$quote$u"]
		set reglig   [list "\\\\oe$sep"  "\\\\OE$sep" "\\\\ae$sep" "\\\\AE$sep"]
		set regoth1  [list "\\\\c$sp[rexp c]|\\\\c{$ws\(c\)$ws}"  "\\\\~[rexp n]"  "\\\\\\$quote[rexp y]"  "\\\\dots$sep"]
		set regcoth1 [list "\\\\c$sp[rexp C]|\\\\c{$ws\(C\)$ws}"  "\\\\~[rexp N]"  "\\\\\\$quote[rexp Y]"]
		set regoth2  [list "\\\\ss$sep"  "\\?`" "\\!`" "\\\\pounds$sep" "\\\\o$sep" "\\\\O$sep" "\\\\S$sep" "\\\\P$sep"]
		set regoth3  [list "{\\\\leavevmode\\\\raise.585ex\\\\hbox{\\\\b{a}}}" "{\\\\leavevmode\\\\raise.6ex\\\\hbox{\\\\b{o}}}" "\\\\accent'27"]
		set regoth4  [list     "\\\\copyright$sep" "\\\\ddag$sep"]
		
		set regall [concat $regas $regcas $reges $regces $regis $regcis $regos $regcos $regus $regcus $reglig $regoth1 $regcoth1 $regoth2 $regoth3 $regoth4]
	}

	############################################################
	set mark {\\\\my-mark-fmd//}
	set space {[ ]+}
	set len [llength $all]
	set i 0
	while {$i < $len} {
		set c [lindex $all $i]
		if {$ww == "1"} {set s [lindex $texall $i]
			regsub -all "$c" $text "$s" text 
		} else {
			set s [lindex $regall $i]
			regsub -all "$s" $text "$mark$c$mark" text
		}
		incr i
	}
	if {$ww == "1"} {   
		regsub -all "$seplater$space" $text "\\\\ " text  
		regsub -all "$seplater\\\\" $text "\\\\" text 
		regsub -all "\($seplater\)\(\\W\)" $text "\\2" text
		regsub -all "$seplater" $text " " text
	} else {
		regsub -all "$mark\\\\ " $text " " text
		regsub -all "$mark" $text "" text
	}
	# workaround Alpha bug
	goto $position
	replaceText $position $endselection $text	
}
